comment
{
  This file contains formulas for fractal type "Attractor". The formulas are constructing objects, they are not actually
  producing fractals, but are generating base objects.
}

geometry(ATTRACTOR)
{
parameter int shape;
parameter real torusRadius,torusOscillationH,torusOscillationV,torusOsciMagn;
parameter real torusArcAngle,torusBodyAngle;
parameter real torusArcAngleAdd,torusBodyAngleAdd;
parameter real torusSpiralLength,torusNumSpirals;
parameter int rotatorShape;
parameter real rotatorDegree,rotatorStartX,rotatorHeight,rotatorRadius,rotatorSphereArc;
parameter bool rotatorClosedShape;
parameter real rotatorPow,rotatorCoeff;
parameter real rotatorOscillationH,rotatorOscillationV,rotatorOsciMagn;
shared double colorIndex;
shared vector zOld;
real h,v,n;
real sinAngle,cosAngle;
real posX,posY,posZ;
real lineEndY,lineEndZ,oscillationFactorH,oscillationFactorV;
real rotatorLength1,rotatorLength2,rotatorLength3;
real kCnt;
real rotHeightMult;
parameter real scaleStart;

	void init(void)
	{
		z=(0,0,0);
		if (shape=="Rotators" && rotatorShape=="Hyperboloid")
		{
			// in case of Rotators/Hyperboloid: Line from (rotatorStartX,0,0) to (0,lineEndY,lineEndZ)
			// i.e.
			lineEndY=cos(2*pi*rotatorDegree/360);
			lineEndZ=sin(2*pi*rotatorDegree/360);
			if (lineEndY==0) {
				rotHeightMult=1;
			} else {
				rotHeightMult=rotatorHeight/lineEndY;
			}
			if (rotatorClosedShape)
			{
				rotatorLength1=rotatorStartX;
				rotatorLength2=sqrt(rotatorStartX*rotatorStartX+lineEndY*lineEndY+lineEndZ*lineEndZ);
				rotatorLength3=sqrt(lineEndY*lineEndY+lineEndZ*lineEndZ);
				// Ok, normalize
				n=rotatorLength1+rotatorLength2+rotatorLength3;
				if (n>0)
				{
					rotatorLength1=rotatorLength1/n;
					rotatorLength2=rotatorLength2/n;
					rotatorLength3=rotatorLength3/n;
				}
			}
		} else if (shape=="Rotators" && rotatorShape=="Paraboloid") {
			rotHeightMult=1;

			if (rotatorClosedShape)
			{
				// length of parabola:
				h = 2.0 * rotatorCoeff;
				v   = sqrt(h^2 + rotatorPow^2);
				rotatorLength2 = v + rotatorPow^2/h*log((h + v)/rotatorPow);

				rotatorLength1=0;
				rotatorLength3=1;
				// Ok, normalize
				n=rotatorLength1+rotatorLength2+rotatorLength3;
				if (n>0)
				{
					rotatorLength1=rotatorLength1/n;
					rotatorLength2=rotatorLength2/n;
					rotatorLength3=rotatorLength3/n;
				}
			}
		} else if (shape=="Rotators" && rotatorShape=="Spheroid") {
			rotHeightMult=1;
			if (rotatorClosedShape)
			{
				// length of arc:
				rotatorLength2 = rotatorRadius*pi*rotatorSphereArc/180;

				rotatorLength1=0;
				posX=sin(rotatorSphereArc/360*2*pi)*rotatorRadius;

				rotatorLength3=sqrt(posX*posX);
				// Ok, normalize
				n=rotatorLength1+rotatorLength2+rotatorLength3;
				if (n>0)
				{
					rotatorLength1=rotatorLength1/n;
					rotatorLength2=rotatorLength2/n;
					rotatorLength3=rotatorLength3/n;
				}
			}
		}
	}
	void loop(void)
	{
		zOld=z;
		h=random.real();
		v=random.real();
		n=random.real();
		if (shape=="Cube")
		{
			n=n*6;
			if (n<1) {
				posX=-1;
				posY=h*2-1;
				posZ=v*2-1;
			} else if (n<2) {
				posX=1;
				posY=h*2-1;
				posZ=v*2-1;
			} else if (n<3) {
				posY=-1;
				posX=h*2-1;
				posZ=v*2-1;
			} else if (n<4) {
				posY=1;
				posX=h*2-1;
				posZ=v*2-1;
			} else if (n<5) {
				posZ=-1;
				posX=h*2-1;
				posY=v*2-1;
			} else {
				posZ=1;
				posX=h*2-1;
				posY=v*2-1;
			}

			// Rotate around y by rv

			z=vector(posX,posY,posZ);
			colorIndex=v;
		}
		else if (shape=="Torus")
		{
			sinAngle=sin(2*pi*torusNumSpirals*torusArcAngle/360*v+torusArcAngleAdd*2*pi/360);
			cosAngle=cos(2*pi*torusNumSpirals*torusArcAngle/360*v+torusArcAngleAdd*2*pi/360);
			posX=1+torusRadius*(1+torusOsciMagn*cos(2*pi*h*torusOscillationH))*(1+torusOsciMagn*cos(2*pi*v*torusOscillationV*torusNumSpirals))*cos(2*pi*torusBodyAngle/360*h+torusBodyAngleAdd*2*pi/360);
			posY=torusRadius*(1+torusOsciMagn*cos(2*pi*h*torusOscillationH))*(1+torusOsciMagn*cos(2*pi*v*torusOscillationV*torusNumSpirals))*sin(2*pi*torusBodyAngle/360*h+torusBodyAngleAdd*2*pi/360);
			posZ=0;

			// Rotate around y by rv

			z=vector(cosAngle*posX-sinAngle*posZ,posY+cos(v*pi)*torusSpiralLength,sinAngle*posX+cosAngle*posZ);
			colorIndex=v;
		} else if (shape=="Rotators") {
			colorIndex=v;
			// there are 3 random numbers, h,v and n between 0 and 1
			// we use v as rotation angle around the axis

			// at first, create a point which in this case gets rotated by an arbitrary amount around the y axis

			if (rotatorShape=="Hyperboloid")
			{
				// line from (rotatorStartX,0,0) to (0,lineEndY,lineEndZ) ==> (rotatorStartX,0,0)+n*(-rotatorStartX,lineEndY,lineEndZ)
				if (rotatorClosedShape) {
					// need the length of the object to see to which segment the point specified by "n" belongs...
					if (n<rotatorLength1) {
						n=n/rotatorLength1;
						posX=n*rotatorStartX;
						posY=0;
						posZ=0;
					} else if (n<rotatorLength1+rotatorLength2) {
						n=(n-rotatorLength1)/rotatorLength2;
						posX=rotatorStartX-n*rotatorStartX;
						posY=n*lineEndY;
						posZ=n*lineEndZ;
					} else {
						n=(n-rotatorLength1-rotatorLength2)/rotatorLength3;
						posX=0;
						posY=lineEndY;
						posZ=n*lineEndZ;
					}
				} else {
					posX=rotatorStartX-n*rotatorStartX;
					posY=n*lineEndY;
					posZ=n*lineEndZ;
				}
			} else if (rotatorShape=="Paraboloid") {
				// creates a parabola in x/y plane
				if (rotatorClosedShape) {
					if (n<rotatorLength1) {
						// cannot be...
						n=n/rotatorLength1;
						posX=n*rotatorStartX;
						posY=0;
						posZ=0;
					} else if (n<rotatorLength1+rotatorLength2) {
						n=(n-rotatorLength1)/rotatorLength2;
						posX=n;
						posY=rotatorCoeff*n^rotatorPow;
						posZ=0;
					} else {
						n=(n-rotatorLength1-rotatorLength2)/rotatorLength3;
						posX=n;
						posY=rotatorCoeff;
						posZ=0;
					}
				} else {
					posX=n;
					posY=rotatorCoeff*n^rotatorPow;
					posZ=0;
				}
				colorIndex=n;
			} else if (rotatorShape=="Spheroid") {
				// creates a spheroid object
				if (rotatorClosedShape) {
					if (n<rotatorLength1) {
						// cannot be...
						posX=0;
						posY=0;
						posZ=0;
					} else if (n<rotatorLength1+rotatorLength2) {
						n=(n-rotatorLength1)/rotatorLength2;
						posX=sin(rotatorSphereArc/360*2*pi*n)*rotatorRadius;
						posY=cos(rotatorSphereArc/360*2*pi*n)*rotatorRadius;
						posZ=0;
					} else {
						n=(n-rotatorLength1-rotatorLength2)/rotatorLength3;

						posX=sin(rotatorSphereArc/360*2*pi*1)*rotatorRadius*n;
						posY=cos(rotatorSphereArc/360*2*pi*1)*rotatorRadius;
						posZ=0;
					}
				} else {
					// at first, create a circle in x/y
					posX=sin(rotatorSphereArc/360*2*pi*n)*rotatorRadius;
					posY=cos(rotatorSphereArc/360*2*pi*n)*rotatorRadius;
					posZ=0;
				}
				colorIndex=n;
			}
			sinAngle=sin(2*pi*v);
			cosAngle=cos(2*pi*v);
			oscillationFactorH=1+sin(2*pi*v*rotatorOscillationH)*rotatorOsciMagn;
			oscillationFactorV=1+sin(2*pi*n*rotatorOscillationV)*rotatorOsciMagn;
			z=vector((cosAngle*posX-sinAngle*posZ)*oscillationFactorH*oscillationFactorV,posY*rotHeightMult*oscillationFactorH*oscillationFactorV,(sinAngle*posX+cosAngle*posZ)*oscillationFactorH*oscillationFactorV);
		}
	}
	bool stop(void)
	{
		return(false);
	}
	void description(void)
	{
		this.title="Geometric Objects";
		this.helpfile = "ChaosPro.chm";
		this.helptopic = "Iteration Formula";
		this.center=(1,3);
		this.angle=0;
		this.maxiter=15000;
		this.periodicity=0;
		this.helpfile="formulas.html";
		shape.caption="Base Shape";
		shape.enum="Cube\nTorus\nRotators";
		shape.default=1;

		torusRadius.caption="Radius";
		torusRadius.visible = (shape=="Torus");
		torusRadius.default=0.3;

		torusArcAngle.caption="Arc Angle";
		torusArcAngle.visible = (shape=="Torus");
		torusArcAngle.default = 360;
		torusArcAngle.min = 0;
		torusArcAngle.max = 360;
		torusArcAngle.hint = "Allows you to render only an arc. Allowed values are from 0 degree to 360 degree (full arc)";

		torusArcAngleAdd.caption="+ Arc Angle";
		torusArcAngleAdd.visible = (shape=="Torus");
		torusArcAngleAdd.default = 90;
		torusArcAngleAdd.hint = "Adds to the arc angle, allowing you to specify which portion of the arc is rendered.";

		torusBodyAngle.caption="Body Angle";
		torusBodyAngle.visible = (shape=="Torus");
		torusBodyAngle.default = 360;
		torusBodyAngle.min = 0;
		torusBodyAngle.max = 360;
		torusBodyAngle.hint = "Allows you to render only a portion of the torus body. May range from 0 degree (nothing is drawn) to 360 degree (full body is drawn)";

		torusBodyAngleAdd.caption="+ Body Angle";
		torusBodyAngleAdd.visible = (shape=="Torus");
		torusBodyAngleAdd.default = 90;
		torusBodyAngleAdd.hint = "Adds to the body angle, allowing you to specify which portion of the body is rendered.";

		torusSpiralLength.caption="Spiral Length";
		torusSpiralLength.visible = (shape=="Torus");
		torusSpiralLength.default=0;
		torusSpiralLength.min=0;
		torusSpiralLength.hint="Let's you specify the length of the spiral.\n\n0 gives a flat spiral, i.e. a torus. Greater values allow to you stretch the spiral.";

		torusNumSpirals.caption="Num Spirals";
		torusNumSpirals.visible = (shape=="Torus");
		torusNumSpirals.default=1;
		torusNumSpirals.min=0;
		torusNumSpirals.hint="Let's you specify the number of spirals.\nAttention: Interfers with Arc Angle. It basically let's you specify how many loops there are.\nSet to 1 if you just want a torus.";

		separator.torusOscillationSettings.caption="Oscillation Setting";
		separator.torusOscillationSettings.visible = (shape=="Torus");

		torusOscillationH.caption="H-Oscillation";
		torusOscillationH.visible = (shape=="Torus");
		torusOscillationH.default=10;
		torusOscillationH.min=0;
		torusOscillationH.hint="Specifies how many oscillations are done horizontally";

		torusOscillationV.caption="V-Oscillation";
		torusOscillationV.visible = (shape=="Torus");
		torusOscillationV.default=20;
		torusOscillationV.min=0;
		torusOscillationV.hint="Specifies how many oscillations are done vertically";

		torusOsciMagn.caption="Oscill. Magnitude";
		torusOsciMagn.visible = (shape=="Torus");
		torusOsciMagn.default=0.1;
		torusOsciMagn.min=0;
		torusOsciMagn.hint="How big is the oscillation? Scale is the same as radius.";

		rotatorShape.caption="Rotator Shape";
		rotatorShape.enum="Cone\nHyperboloid\nParaboloid\nSpheroid";
		rotatorShape.hint = "A rotator starts with a 2D curve and rotates it around the y axis.\nHyperboloid starts with a line segment";
		rotatorShape.visible = (shape=="Rotators");
		rotatorShape.default=1;

		rotatorStartX.caption="Starting point (X)";
		rotatorStartX.visible = (shape=="Rotators" && rotatorShape=="Hyperboloid");
		rotatorStartX.hint = "Starting point at (x,0,0) for the line segment";
		rotatorStartX.default=1;

		rotatorDegree.caption="Degree";
		rotatorDegree.default=0;
		rotatorDegree.hint="Specifies the end point of the line segment which then gets rotated: Rotates the point (0,1,0) by this value (in degree). 0 leads to a cone, 45 leads to a vase, etc.";
		rotatorDegree.visible = (shape=="Rotators" && rotatorShape=="Hyperboloid");

		rotatorCoeff.caption="Coefficient";
		rotatorCoeff.hint="Given a general parabola formula:\n\n   y = a * x^p\n\nthe coefficient here specifies a.";
		rotatorCoeff.visible=(shape=="Rotators" && rotatorShape=="Paraboloid");
		rotatorCoeff.default=1;

		rotatorPow.caption="Pow";
		rotatorPow.visible=(shape=="Rotators" && rotatorShape=="Paraboloid");
		rotatorPow.hint="Given a general parabola formula:\n\n   y = a * x^p\n\nthe pow specifies p.";
		rotatorPow.default=2;

		rotatorSphereArc.caption="Arc of Sphere";
		rotatorSphereArc.visible = (shape=="Rotators" && rotatorShape=="Spheroid");
		rotatorSphereArc.default=360;

		rotatorRadius.caption="Radius";
		rotatorRadius.visible = (shape=="Rotators" && rotatorShape=="Spheroid");
		rotatorRadius.default=1;


		rotatorHeight.caption="Height";
		rotatorHeight.visible = (shape=="Rotators" && rotatorShape=="Hyperboloid");
		rotatorHeight.default=1;

		rotatorClosedShape.caption="Closed Shape";
		rotatorClosedShape.hint = "If enabled, produces a closed shape.";
		rotatorClosedShape.visible = (shape=="Rotators");
		rotatorClosedShape.default=false;

		rotatorOscillationH.caption="H-Oscillation";
		rotatorOscillationH.visible = (shape=="Rotators");
		rotatorOscillationH.default=10;
		rotatorOscillationH.min=0;
		rotatorOscillationH.hint="Specifies how many oscillations are done horizontally";

		rotatorOscillationV.caption="V-Oscillation";
		rotatorOscillationV.visible = (shape=="Rotators");
		rotatorOscillationV.default=5;
		rotatorOscillationV.min=0;
		rotatorOscillationV.hint="Specifies how many oscillations are done vertically";

		rotatorOsciMagn.caption="Oscill. Magnitude";
		rotatorOsciMagn.visible = (shape=="Rotators");
		rotatorOsciMagn.default=0.1;
		rotatorOsciMagn.min=0;
		rotatorOsciMagn.hint="How big is the oscillation? Scale is the same as radius.";

	}
}



unitcube(ATTRACTOR)
{
real h,v,n;
real posX,posY,posZ;
parameter bool useWireframe;
parameter int numPieces;
parameter bool showSide1,showSide2,showSide3,showSide4,showSide5,showSide6;

	void init(void)
	{
		// empty
	}
	void loop(void)
	{
		h=random.real();
		v=random.real();
		n=random.real()*6;
		if (useWireframe) {
			if (random.real()<0.5) {
				h=floor(h*(numPieces+1))/numPieces;
			} else {
				v=floor(v*(numPieces+1))/numPieces;
			}
		}
		if (n<1) {
			if (showSide1) {
				posX=-1;
				posY=h*2-1;
				posZ=v*2-1;
			} else {
				hiddenPixel=true;
			}
		} else if (n<2) {
			if (showSide2) {
				posX=1;
				posY=h*2-1;
				posZ=v*2-1;
			} else {
				hiddenPixel=true;
			}
		} else if (n<3) {
			if (showSide3) {
				posY=-1;
				posX=h*2-1;
				posZ=v*2-1;
			} else {
				hiddenPixel=true;
			}
		} else if (n<4) {
			if (showSide4) {
				posY=1;
				posX=h*2-1;
				posZ=v*2-1;
			} else {
				hiddenPixel=true;
			}
		} else if (n<5) {
			if (showSide5) {
				posZ=-1;
				posX=h*2-1;
				posY=v*2-1;
			} else {
				hiddenPixel=true;
			}
		} else {
			if (showSide6) {
				posZ=1;
				posX=h*2-1;
				posY=v*2-1;
			} else {
				hiddenPixel=true;
			}
		}
		z=vector(posX/2,posY/2,posZ/2);
	}
	void description(void)
	{
		this.title="Unit Cube";
		this.helpfile = "ChaosPro.chm";
		this.helptopic = "Iteration Formula";
		
		useWireframe.caption="Wireframe";
		useWireframe.default=false;
		
		numPieces.caption="Pieces";
		numPieces.default=20;
		numPieces.min=1;
		numPieces.max=1E10;
		numPieces.visible=(useWireframe);
		
		showSide1.caption="Show Side 1";
		showSide1.default=true;

		showSide2.caption="Show Side 2";
		showSide2.default=true;

		showSide3.caption="Show Side 3";
		showSide3.default=true;

		showSide4.caption="Show Side 4";
		showSide4.default=true;

		showSide5.caption="Show Side 5";
		showSide5.default=true;

		showSide6.caption="Show Side 6";
		showSide6.default=true;
		
	}
}

sphere(ATTRACTOR)
{
parameter real radius;
parameter real thickness;
parameter int shape;
parameter vector obj_center;
real h,v,r;

	void loop(void)
	{
		h=2*pi*random.real();
		v=pi*random.real();
		r = random.real()*thickness;
		z=obj_center+vector((radius-r)*cos(h)*sin(v),(radius-r)*sin(h)*sin(v),(radius-r)*cos(v));
	}
	bool stop(void)
	{
		return(false);
	}
	void description(void)
	{
		this.title="Sphere";
		this.helpfile = "ChaosPro.chm";
		this.helptopic = "Iteration Formula";
		obj_center.caption="Center";
		obj_center.default=(0,0,0);
		radius.caption="Radius";
		radius.default=1;

		thickness.caption="Thickness";
		thickness.default=0;
		thickness.min=0;
		thickness.max=1;
	}
}


Kleinbottle(ATTRACTOR)
{
parameter real a;
parameter real thickness;
parameter int shape;
parameter vector obj_center;
real u,v,tx,ty,tz;
shared double colorIndex;

	void init(void)
	{
	}
	void loop(void)
	{
		u=random.real();
		v=random.real();
		if (u*100-floor(u*100)<0.2) {
			colorIndex=0;
		} else if (v*50-floor(v*50)<0.2) {
			colorIndex=0;
		} else {
			colorIndex=0.9;
		}

		u=(u-0.5)*2*pi;
		v=(v-0.5)*2*pi;

		tx = cos(u)*(a + sin(v)*cos(u/2) - sin(2*v)*sin(u/2)/2);
		ty = sin(u)*(a + sin(v)*cos(u/2) - sin(2*v)*sin(u/2)/2);
		tz = sin(u/2)*sin(v) + cos(u/2)*sin(2*v)/2;
		z=vector(tx,ty,tz);
	}
	void description(void)
	{
		this.title="Klein bottle";
		this.helpfile = "ChaosPro.chm";
		this.helptopic = "Iteration Formula";

		a.caption="Size";
		a.default=1;
		a.min=-1E10;
		a.max=1E10;
		a.randomizable=true;
		a.randomizeMin=0;
		a.randomizeMax=10;
	}
}

Sphericalharmonics(ATTRACTOR)
{
parameter vector obj_center;
parameter int variableType;
parameter real m0,m1,m2,m3,m4,m5,m6,m7;
int im0,im1,im2,im3,im4,im5,im6,im7;
real rm0,rm1,rm2,rm3,rm4,rm5,rm6,rm7;
real h,v,r;

	void init(void)
	{
		// empty...
		if (variableType=="Int") {
			im0=m0;
			im1=m1;
			im2=m2;
			im3=m3;
			im4=m4;
			im5=m5;
			im6=m6;
			im7=m7;
		} else {
			rm0=m0;
			rm1=m1;
			rm2=m2;
			rm3=m3;
			rm4=m4;
			rm5=m5;
			rm6=m6;
			rm7=m7;
		}
	}
	void loop(void)
	{
		h=2*pi*random.real();
		v=pi*random.real();
		r=0;
		if (variableType=="Int") {
			r = r+ sin(im0*v)^im1;
			r = r+ cos(im2*v)^im3;
			r = r+ sin(im4*h)^im5;
			r = r+ cos(im6*h)^im7;
		} else {
			r = r+ sin(rm0*v)^rm1;
			r = r+ cos(rm2*v)^rm3;
			r = r+ sin(rm4*h)^rm5;
			r = r+ cos(rm6*h)^rm7;
		}



		z=obj_center+vector(r*cos(h)*sin(v),r*sin(h)*sin(v),r*cos(v));
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Spherical Harmonics";
		this.helpfile = "ChaosPro.chm";
		this.helptopic = "Iteration Formula";
		obj_center.caption="Center";
		obj_center.default=(0,0,0);

		variableType.caption="Variable Type";
		variableType.enum="Int\nReal";
		variableType.hint="Mathematically only int values are allowed,\nbut real values also produce interesting results";
		variableType.default=0;

		m0.caption="M0";
		m0.default=0;
		m0.randomizable=true;
		m0.min=0;
		m0.max=1E10;
		m0.randomizeMin=0;
		m0.randomizeMax=10;

		m1.caption="M1";
		m1.default=0;
		m1.randomizable=true;
		m1.min=0;
		m1.max=1E10;
		m1.randomizeMin=0;
		m1.randomizeMax=10;

		m2.caption="M2";
		m2.default=0;
		m2.randomizable=true;
		m2.min=0;
		m2.max=1E10;
		m2.randomizeMin=0;
		m2.randomizeMax=10;

		m3.caption="M3";
		m3.default=0;
		m3.randomizable=true;
		m3.min=0;
		m3.max=1E10;
		m3.randomizeMin=0;
		m3.randomizeMax=10;

		m4.caption="M4";
		m4.default=0;
		m4.randomizable=true;
		m4.min=0;
		m4.max=1E10;
		m4.randomizeMin=0;
		m4.randomizeMax=10;

		m5.caption="M5";
		m5.default=0;
		m5.randomizable=true;
		m5.min=0;
		m5.max=1E10;
		m5.randomizeMin=0;
		m5.randomizeMax=10;

		m6.caption="M6";
		m6.default=0;
		m6.randomizable=true;
		m6.min=0;
		m6.max=1E10;
		m6.randomizeMin=0;
		m6.randomizeMax=10;

		m7.caption="M7";
		m7.default=0;
		m7.randomizable=true;
		m7.min=0;
		m7.max=1E10;
		m7.randomizeMin=0;
		m7.randomizeMax=10;
	}
}

Supershape(ATTRACTOR)
{
parameter vector obj_center;

real phi,theta,r1,r2;
parameter real n11,n12,n13,n21,n22,n23,m1,m2,a1,b1,a2,b2;
real tx,ty,tz;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		phi=(random.real()-0.5)*pi;
		theta=(random.real()-0.5)*2*pi;
		// n1,n2,n3,a,b,m
		r1=(cabs(1/a1*cos(m1*theta/4))^n12+cabs(1/b1*sin(m1*theta/4))^n13)^(-1/n11);
		r2=(cabs(1/a2*cos(m2*phi/4))^n22+cabs(1/b2*sin(m2*phi/4))^n23)^(-1/n21);

		tx=r1*cos(theta)*r2*cos(phi);
		ty=r1*sin(theta)*r2*cos(phi);
		tz=r2*sin(phi);

		z=obj_center+vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Supershape";
		this.helpfile = "ChaosPro.chm";
		this.helptopic = "Iteration Formula";
		obj_center.caption="Center";
		obj_center.default=(0,0,0);


		separator.label1.caption="Supershape 1";
		m1.caption="M";
		m1.default=0.75;
		m1.randomizable=true;
		m1.min=0;
		m1.max=1E10;
		m1.randomizeMin=0;
		m1.randomizeMax=10;

		n11.caption="n1";
		n11.default=6;
		n11.randomizable=true;
		n11.min=-1E10;
		n11.max=1E10;
		n11.randomizeMin=0;
		n11.randomizeMax=10;

		n12.caption="n2";
		n12.default=2;
		n12.randomizable=true;
		n12.min=-1E10;
		n12.max=1E10;
		n12.randomizeMin=0;
		n12.randomizeMax=10;

		n13.caption="n3";
		n13.default=5;
		n13.randomizable=true;
		n13.min=-1E10;
		n13.max=1E10;
		n13.randomizeMin=0;
		n13.randomizeMax=10;

		a1.caption="a";
		a1.default=4;
		a1.randomizable=true;
		a1.min=0;
		a1.max=1E10;
		a1.randomizeMin=0;
		a1.randomizeMax=10;

		b1.caption="b";
		b1.default=2;
		b1.randomizable=true;
		b1.min=0;
		b1.max=1E10;
		b1.randomizeMin=0;
		b1.randomizeMax=10;

		separator.label2.caption="Supershape 2";

		m2.caption="M";
		m2.default=9;
		m2.randomizable=true;
		m2.min=0;
		m2.max=1E10;
		m2.randomizeMin=0;
		m2.randomizeMax=10;

		n21.caption="n1";
		n21.default=8;
		n21.randomizable=true;
		n21.min=-1E10;
		n21.max=1E10;
		n21.randomizeMin=0;
		n21.randomizeMax=10;

		n22.caption="n2";
		n22.default=6;
		n22.randomizable=true;
		n22.min=-1E10;
		n22.max=1E10;
		n22.randomizeMin=0;
		n22.randomizeMax=10;

		n23.caption="n3";
		n23.default=7;
		n23.randomizable=true;
		n23.min=-1E10;
		n23.max=1E10;
		n23.randomizeMin=0;
		n23.randomizeMax=10;

		a2.caption="a";
		a2.default=4;
		a2.randomizable=true;
		a2.min=0;
		a2.max=1E10;
		a2.randomizeMin=0;
		a2.randomizeMax=10;

		b2.caption="b";
		b2.default=7;
		b2.randomizable=true;
		b2.min=0;
		b2.max=1E10;
		b2.randomizeMin=0;
		b2.randomizeMax=10;
	}
}

KuenSurface(ATTRACTOR)
{
real cosU,cosV,sinU,sinV,divisor;
real u,v;
real tx,ty,tz;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=(random.real()*9)-4.5;
		v=random.real()*pi;
		cosU=cos(u);
		cosV=cos(v);
		sinU=sin(u);
		sinV=sin(v);
		divisor=1+u*u*sinV*sinV;

		tx = 2*(cosU+u*sinU)*sinV/divisor;
		ty = 2*(sinU-u*cosU)*sinV/divisor;
		tz = ln(tan(0.5*v))+2*cosV/divisor;
		if (cabs(tz)>2) {
			hiddenPixel=true;
		}
		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Kuen Surface";
		this.helpfile = "ChaosPro.chm";
		this.helptopic = "Iteration Formula";
	}
}

BohemianDome(ATTRACTOR)
{
real u,v;
real tx,ty,tz;
parameter real a,b,c;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=random.real()*2*pi;
		v=random.real()*2*pi;
		tx = a*cos(u);
		ty = b*cos(v) + a*sin(u);
		tz = c*sin(v);

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Bohemian Dome";

		a.caption="Width 1";
		a.default=1;
		a.randomizable=true;
		a.min=0;
		a.max=1E10;
		a.randomizeMin=0;
		a.randomizeMax=10;

		b.caption="Width 2";
		b.default=1;
		b.randomizable=true;
		b.min=0;
		b.max=1E10;
		b.randomizeMin=0;
		b.randomizeMax=10;

		c.caption="Height";
		c.default=1;
		c.randomizable=true;
		c.min=0;
		c.max=1E10;
		c.randomizeMin=0;
		c.randomizeMax=10;

	}
}

Cornucopia(ATTRACTOR)
{
real u,v;
real tx,ty,tz;
parameter real a,b,t2;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=(random.real()*2*pi-pi);
		v=(random.real()*2*pi)*t2;

		tx = e^(b*0.1*v)*cos(v) + e^(a*0.1*v)*cos(u)*cos(v);
		ty = e^(b*0.1*v)*sin(v) + e^(a*0.1*v)*cos(u)*sin(v);
		tz = e^(a*0.1*v)*sin(u);

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Cornucopia";

		t2.caption="Lenght";
		t2.default=2;
		t2.randomizable=true;
		t2.min=0;
		t2.max=1E10;
		t2.randomizeMin=0;
		t2.randomizeMax=10;

		a.caption="a";
		a.default=0.1;
		a.randomizable=true;
		a.min=-1E10;
		a.max=1E10;
		a.randomizeMin=0;
		a.randomizeMax=2;

		b.caption="b";
		b.default=1;
		b.randomizable=true;
		b.min=-1E10;
		b.max=1E10;
		b.randomizeMin=0;
		b.randomizeMax=2;
	}
}

DinisCup(ATTRACTOR)
{
real u,v;
real tx,ty,tz;
parameter real a,b;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=random.real()*4*pi;
		v=random.real()*2+0.001;

		tx = a*cos(u)*sin(v);
		ty = a*sin(u)*sin(v);
		tz = a*(cos(v)+ln(tan(v*0.5)))+b*u;

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Dini's Cup";

		a.caption="a";
		a.default=1;
		a.randomizable=true;
		a.min=-1E10;
		a.max=1E10;
		a.randomizeMin=0;
		a.randomizeMax=10;

		b.caption="b";
		b.default=0.1;
		b.randomizable=true;
		b.min=-1E10;
		b.max=1E10;
		b.randomizeMin=0;
		b.randomizeMax=10;
	}
}

Seashell(ATTRACTOR)
{
real u,v;
real tx,ty,tz,h;
parameter real t1,a,b,c,n;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=(random.real()*2);
		v=(random.real()*2);

		h = 1-0.5*v;

		tx = a*h*cos(n*v*pi)*(1 + cos(u*pi)) + c*cos(n*v*pi);
		ty = a*h*sin(n*v*pi)*(1 + cos(u*pi)) + c*sin(n*v*pi);
		tz = b*0.5*v + a*h*sin(u*pi);

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Seashell";

		t1.caption="Lenght";
		t1.default=1;
		t1.randomizable=true;
		t1.min=0;
		t1.max=1E10;
		t1.randomizeMin=0;
		t1.randomizeMax=10;

		a.caption="Width";
		a.default=1;
		a.randomizable=true;
		a.min=-1E10;
		a.max=1E10;
		a.randomizeMin=0;
		a.randomizeMax=1;

		b.caption="Height";
		b.default=2;
		b.randomizable=true;
		b.min=-1E10;
		b.max=1E10;
		b.randomizeMin=0;
		b.randomizeMax=10;

		c.caption="Inner Radius";
		c.default=0;
		c.randomizable=true;
		c.min=-1E10;
		c.max=1E10;
		c.randomizeMin=0;
		c.randomizeMax=1;

		n.caption="Num. Spirals";
		n.default=3;
		n.randomizable=true;
		n.min=-1E10;
		n.max=1E10;
		n.randomizeMin=0;
		n.randomizeMax=10;
	}
}

SineSurface(ATTRACTOR)
{
real u,v;
real tx,ty,tz,h;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=(random.real()*2*pi)-pi;
		v=(random.real()*2*pi)-pi;

		tx = sin(u);
		ty = sin(v);
		tz = sin(u+v);

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Sine Surface";
	}
}

SteinbachScrew(ATTRACTOR)
{
real u,v;
real tx,ty,tz,h;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=(random.real()*8)-4;
		v=(random.real()*2*pi);

		tx = u*cos(v);
		ty = u*sin(v);
		tz = v*cos(u);

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Steinbach Screw";
	}
}

HyperbolicHelicoid(ATTRACTOR)
{
real u,v;
real tx,ty,tz,h;
parameter real a,t1,t2;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=(random.real()*8)-4;
		v=(random.real()*8)-4;
		u=u*t1;
		v=v*t2;

		tx = sinh(v)*cos(a*u)/(1+cosh(u)*cosh(v));
		ty = sinh(v)*sin(a*u)/(1+cosh(u)*cosh(v));
		tz = cosh(v)*sinh(u)/(1+cosh(u)*cosh(v));

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Hyperbolic Helicoid";

		t1.caption="Lenght 1";
		t1.default=1;
		t1.randomizable=true;
		t1.min=0;
		t1.max=1E10;
		t1.randomizeMin=0;
		t1.randomizeMax=1;

		t2.caption="Lenght 2";
		t2.default=3;
		t2.randomizable=true;
		t2.min=0;
		t2.max=1E10;
		t2.randomizeMin=0;
		t2.randomizeMax=10;

		a.caption="Width";
		a.default=3;
		a.randomizable=true;
		a.min=-1E10;
		a.max=1E10;
		a.randomizeMin=0;
		a.randomizeMax=10;
	}
}

MaederOwl(ATTRACTOR)
{
real u,v;
real tx,ty,tz,h;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=random.real()*4*pi;
		v=(random.real())+0.001;

		tx = v*cos(u)-0.5*v^2*cos(2*u);
		ty = -v*sin(u) - 0.5*v^2*sin(2*u);
		tz = 4*exp(1.5*log(v))*cos(3*u/2)/3;

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Maeders Owl";
	}
}

TranguloidTrefoil(ATTRACTOR)
{
real u,v;
real tx,ty,tz,h;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=(random.real()*2*pi)-pi;
		v=(random.real()*2*pi)-pi;

		tx = 2*sin(3*u)/(2 + cos(v));
		ty = 2*(sin(u) + 2*sin(2*u))/(2 + cos(v + 2*pi/3));
		tz = (cos(u) - 2*cos(2*u))*(2 + cos(v))*(2 + cos(v + 2*pi/3))/4;

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Tranguloid Trefoil";
	}
}


Wavesphere(ATTRACTOR)
{
real u,v;
real tx,ty,tz,h;
parameter real t1,t2;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=(random.real()*14.5)*t1;
		v=(random.real()*2*pi)*t2;

		tx = u*cos(cos(u))*cos(v);
		ty = u*cos(cos(u))*sin(v);
		tz = u*sin(cos(u));

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Wave Sphere";

		t1.caption="Lenght 1";
		t1.default=1;
		t1.randomizable=true;
		t1.min=0;
		t1.max=1E10;
		t1.randomizeMin=0;
		t1.randomizeMax=1;

		t2.caption="Lenght 2";
		t2.default=0.8;
		t2.randomizable=true;
		t2.min=0;
		t2.max=1E10;
		t2.randomizeMin=0;
		t2.randomizeMax=10;
	}
}

SnailsShells(ATTRACTOR)
{
real u,v;
real tx,ty,tz;

real a,b,c,h,k,w;
real uLen,uMin;
int R;

parameter real paramA,paramB,paramC,paramH,paramK,paramW;
parameter real paramULen,paramUMin;
parameter int paramR;
parameter int snailType;

	void init(void)
	{
		if (snailType=="Pseudoheliceras subcatenatum") {
			a=1.6;
			b=1.6;
			c=1.0;
			h=1.5;
			k=-7.0;
			w=0.075;
			uMin=-50;
			uLen=49;
			R=1;
		} else if (snailType=="Astroceras") {
			a=1.25;
			b=1.25;
			c=1.0;
			h=3.5;
			k=0;
			w=0.12;
			uMin=-40;
			uLen=49;
			R=1;
		} else if (snailType=="Bellerophina") {
			a=0.85;
			b=1.2;
			c=1.0;
			h=0.75;
			k=0;
			w=0.06;
			uMin=-10;
			uLen=9;
			R=1;
		}  else if (snailType=="Euhoplites") {
			a=0.6;
			b=0.4;
			c=1.0;
			h=0.9;
			k=0;
			w=0.1626;
			uMin=-40;
			uLen=39;
			R=1;
		} else if (snailType=="Nautilus") {
			a=1.0;
			b=0.6;
			c=1.0;
			h=1.0;
			k=0;
			w=0.18;
			uMin=-20;
			uLen=21;
			R=1;
		} else if (snailType=="Natica stellata") {
			a=2.6;
			b=2.4;
			c=1.0;
			h=1.25;
			k=-2.8;
			w=0.18;
			uMin=-20;
			uLen=21;
			R=1;
		} else if (snailType=="Mya arenaria") {
			a=0.85;
			b=1.6;
			c=3.0;
			h=0.9;
			k=0;
			w=2.5;
			uMin=-1.0;
			uLen=1.52;
			R=1;
		} else {
			a=paramA;
			b=paramB;
			c=paramC;
			h=paramH;
			k=paramK;
			w=paramW;
			uMin=paramUMin;
			uLen=paramULen;
		}
		R=1;
	}
	void loop(void)
	{
		u=random.real()*(uLen)+uMin;
		v=(random.real()*2*pi);

		tx = exp(w*u)*(h + a*cos(v))*cos(c*u);
		ty = R*exp(w*u)*(h + a*cos(v))*sin(c*u);
		tz = exp(w*u)*(k + b*sin(v));

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Snails and Shells";

		snailType.caption="Snailtype";
		snailType.enum="Pseudoheliceras subcatenatum\nAstroceras\nBellerophina\nEuhoplites\nNautilus\nNatica stellata\nMya arenaria\nArbitrary";

		paramA.caption="a";
		paramA.default=1.6;
		paramA.randomizable=true;
		paramA.min=0;
		paramA.max=1E10;
		paramA.randomizeMin=0;
		paramA.randomizeMax=2;
		paramA.visible=(snailType=="Arbitrary");

		paramB.caption="b";
		paramB.default=1.6;
		paramB.randomizable=true;
		paramB.min=0;
		paramB.max=1E10;
		paramB.randomizeMin=0;
		paramB.randomizeMax=2;
		paramB.visible=(snailType=="Arbitrary");

		paramC.caption="c";
		paramC.default=1.6;
		paramC.randomizable=true;
		paramC.min=0;
		paramC.max=1E10;
		paramC.randomizeMin=0;
		paramC.randomizeMax=2;
		paramC.visible=(snailType=="Arbitrary");

		paramH.caption="h";
		paramH.default=1.6;
		paramH.randomizable=true;
		paramH.min=0;
		paramH.max=1E10;
		paramH.randomizeMin=0;
		paramH.randomizeMax=2;
		paramH.visible=(snailType=="Arbitrary");

		paramK.caption="k";
		paramK.default=1.6;
		paramK.randomizable=true;
		paramK.min=-1E10;
		paramK.max=1E10;
		paramK.randomizeMin=-10;
		paramK.randomizeMax=2;
		paramK.visible=(snailType=="Arbitrary");

		paramW.caption="w";
		paramW.default=1.6;
		paramW.randomizable=true;
		paramW.min=0;
		paramW.max=1E10;
		paramW.randomizeMin=0;
		paramW.randomizeMax=2;
		paramW.visible=(snailType=="Arbitrary");

		paramUMin.caption="u Minimum";
		paramUMin.default=-50;
		paramUMin.randomizable=true;
		paramUMin.min=-1E10;
		paramUMin.max=1E10;
		paramUMin.randomizeMin=-50;
		paramUMin.randomizeMax=50;
		paramUMin.visible=(snailType=="Arbitrary");

		paramULen.caption="u Range";
		paramULen.default=49;
		paramULen.randomizable=true;
		paramULen.min=0;
		paramULen.max=1E10;
		paramULen.randomizeMin=0;
		paramULen.randomizeMax=100;
		paramULen.visible=(snailType=="Arbitrary");
	}
}

Quartic(ATTRACTOR)
{
real u,v;
real tx,ty,tz,h;
parameter real t1,t2;

parameter real a1,a2,a3,a4,a5,a6,a7,a8,a9,a10,a11,a12,a13,a14,a15,a16,a17,a18,a19,a20,a21,a22,a23,a24,a25,a26,a27,a28,a29,a30,a31,a32,a33,a34,a35;
real vx,vy,vz;
real A,B,C,D,E;
real alpha,beta,gamma;
real l1,l2,l3,l4,l;
real r1,r2,r3,r4;
real P,Q,R,U,W;
real y;
real s1,s2;
int numSolutions,prevSolution;

	void init(void)
	{
		// empty...
		prevSolution=0;
		numSolutions=0;
	}
	void loop(void)
	{
		if (prevSolution==0 || prevSolution>=numSolutions) {
			// beliebigen Punkt auf einer Einheitskugel erzeugen:
			h=2*pi*random.real();
			v=pi*random.real();
			vx=cos(h)*sin(v);
			vy=sin(h)*sin(v);
			vz=cos(v);
			// Gerade definiert durch: (x,y,z)=l*(vx,vy,vz)+(px,py,pz)
			A=a1*vx^4+a2*vx^3*vy+a3*vx^3*vz +a5*vx^2*vy^2+a6*vx^2*vy*vz+a8*vx^2*vz^2+a11*vx*vy^3+a12*vx*vy^2*vz+a14*vx*vy*vz^2+a17*vx*vz^3+a21*vy^4+a22*vy^3*vz+a24*vy^2*vz^2+a27*vy*vz^3+a31*vz^4;
			B=a4*vx^3+a7*vx^2*vy+a9*vx^2*vz+a13*vx*vy^2+a15*vx*vy*vz+a18*vx*vz^2+a23*vy^3+a25*vy^2*vz+a28*vy*vz^2+a32*vz^3;
			C=a10*vx^2+a16*vx*vy+a19*vx*vz+a26*vy^2+a29*vy*vz+a33*vz^2;
			D=a20*vx+a30*vy+a34*vz;
			E=a35;
			alpha=-(3*sqr(B))/(8*sqr(A))+C/A;
			beta=(B^3)/(8*A^3)-B*C/(2*A^2)+D/A;
			gamma=-(3*B^4)/(256*A^4)+C*(B^2)/(16*A^3)-B*D/(4*A^2)+E/A;
			if (beta==0) {
				r1=sqrt(alpha^2-4*gamma);
				l1=-B/(4*A)+sqrt((-alpha+r1)*0.5);
				l2=-B/(4*A)+sqrt((-alpha-r1)*0.5);
				l3=-B/(4*A)-sqrt((-alpha+r1)*0.5);
				l4=-B/(4*A)-sqrt((-alpha-r1)*0.5);
				// hiddenPixel=true;
			} else {
				// 
				P = -(alpha^2)/12-gamma;
				Q = -(alpha^3)/108+alpha*gamma/3-(beta^2)/8;
				R=-Q/2+sqrt(Q^2/4+(P^3)/27);
				U=R^(1/3);
				if (U==0) {
					y=-5/6*alpha+U-Q^(1/3);
				} else {
					y=-5/6*alpha+U-P/(3*U);
				}
				W=sqrt(alpha+2*y);
				s1=-(3*alpha+2*y+2*beta/W);
				s2=-(3*alpha+2*y-2*beta/W);
				if (s1>=0 && s2>=0) {
					// 4 Loesungen
					s1=sqrt(s1);
					s2=sqrt(s2);
					l1=-B/(4*A) + 0.5*( W + s1);
					l2=-B/(4*A) + 0.5*( W - s1);
					l3=-B/(4*A) + 0.5*( -W + s2);
					l4=-B/(4*A) + 0.5*( -W - s2);
					numSolutions=4;
				} else if (s1>=0) {
					s1=sqrt(s1);
					l1=-B/(4*A) + 0.5*( W + s1);
					l2=-B/(4*A) + 0.5*( W - s1);
					numSolutions=2;
				} else if (s2>=0) {
					s2=sqrt(s2);
					l1=-B/(4*A) + 0.5*( -W + s2);
					l2=-B/(4*A) + 0.5*( -W - s2);
					numSolutions=2;
				} else {
					hiddenPixel=true;
					numSolutions=0;
				}
			}
			prevSolution=1;
		} else {
			prevSolution=prevSolution+1;
		}
		if (prevSolution==1) {
			l=l1;
		} else if (prevSolution==2) {
			l=l2;
		} else if (prevSolution==3) {
			l=l3;
		} else if (prevSolution==4) {
			l=l4;
		}
		
		z=vector(l*vx,l*vy,l*vz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Quartic";

		a1.caption="a1";
		a1.default=1;
		a1.randomizable=true;
		a1.min=-1E10;
		a1.max=1E10;
		a1.randomizeMin=-2;
		a1.randomizeMax=2;

		a2.caption="a2";
		a2.default=0;
		a2.randomizable=true;
		a2.min=-1E10;
		a2.max=1E10;
		a2.randomizeMin=-2;
		a2.randomizeMax=2;

		a3.caption="a3";
		a3.default=0;
		a3.randomizable=true;
		a3.min=-1E11;
		a3.max=1E11;
		a3.randomizeMin=-2;
		a3.randomizeMax=2;

		a4.caption="a4";
		a4.default=0;
		a4.randomizable=true;
		a4.min=-1E11;
		a4.max=1E11;
		a4.randomizeMin=-2;
		a4.randomizeMax=2;

		a5.caption="a5";
		a5.default=1;
		a5.randomizable=true;
		a5.min=-1E11;
		a5.max=1E11;
		a5.randomizeMin=-2;
		a5.randomizeMax=2;

		a6.caption="a6";
		a6.default=0;
		a6.randomizable=true;
		a6.min=-1E12;
		a6.max=1E12;
		a6.randomizeMin=-2;
		a6.randomizeMax=2;

		a7.caption="a7";
		a7.default=4;
		a7.randomizable=true;
		a7.min=-1E12;
		a7.max=1E12;
		a7.randomizeMin=-2;
		a7.randomizeMax=2;

		a8.caption="a8";
		a8.default=2;
		a8.randomizable=true;
		a8.min=-1E12;
		a8.max=1E12;
		a8.randomizeMin=-2;
		a8.randomizeMax=2;

		a9.caption="a9";
		a9.default=0;
		a9.randomizable=true;
		a9.min=-1E13;
		a9.max=1E13;
		a9.randomizeMin=-2;
		a9.randomizeMax=2;

		a10.caption="a10";
		a10.default=-2;
		a10.randomizable=true;
		a10.min=-1E13;
		a10.max=1E13;
		a10.randomizeMin=-2;
		a10.randomizeMax=2;

		a11.caption="a11";
		a11.default=0;
		a11.randomizable=true;
		a11.min=-1E13;
		a11.max=1E13;
		a11.randomizeMin=-2;
		a11.randomizeMax=2;

		a12.caption="a12";
		a12.default=0;
		a12.randomizable=true;
		a12.min=-1E14;
		a12.max=1E14;
		a12.randomizeMin=-2;
		a12.randomizeMax=2;

		a13.caption="a13";
		a13.default=0;
		a13.randomizable=true;
		a13.min=-1E14;
		a13.max=1E14;
		a13.randomizeMin=-2;
		a13.randomizeMax=2;

		a14.caption="a14";
		a14.default=0;
		a14.randomizable=true;
		a14.min=-1E14;
		a14.max=1E14;
		a14.randomizeMin=-2;
		a14.randomizeMax=2;

		a15.caption="a15";
		a15.default=0;
		a15.randomizable=true;
		a15.min=-1E15;
		a15.max=1E15;
		a15.randomizeMin=-2;
		a15.randomizeMax=2;

		a16.caption="a16";
		a16.default=0;
		a16.randomizable=true;
		a16.min=-1E15;
		a16.max=1E15;
		a16.randomizeMin=-2;
		a16.randomizeMax=2;

		a17.caption="a17";
		a17.default=0;
		a17.randomizable=true;
		a17.min=-1E15;
		a17.max=1E15;
		a17.randomizeMin=-2;
		a17.randomizeMax=2;

		a18.caption="a18";
		a18.default=0;
		a18.randomizable=true;
		a18.min=-1E16;
		a18.max=1E16;
		a18.randomizeMin=-2;
		a18.randomizeMax=2;

		a19.caption="a19";
		a19.default=0;
		a19.randomizable=true;
		a19.min=-1E16;
		a19.max=1E16;
		a19.randomizeMin=-2;
		a19.randomizeMax=2;

		a20.caption="a20";
		a20.default=0;
		a20.randomizable=true;
		a20.min=-1E16;
		a20.max=1E16;
		a20.randomizeMin=-2;
		a20.randomizeMax=2;

		a21.caption="a21";
		a21.default=0;
		a21.randomizable=true;
		a21.min=-1E17;
		a21.max=1E17;
		a21.randomizeMin=-2;
		a21.randomizeMax=2;

		a22.caption="a22";
		a22.default=0;
		a22.randomizable=true;
		a22.min=-1E17;
		a22.max=1E17;
		a22.randomizeMin=-2;
		a22.randomizeMax=2;

		a23.caption="a23";
		a23.default=0;
		a23.randomizable=true;
		a23.min=-1E17;
		a23.max=1E17;
		a23.randomizeMin=-2;
		a23.randomizeMax=2;

		a24.caption="a24";
		a24.default=1;
		a24.randomizable=true;
		a24.min=-1E18;
		a24.max=1E18;
		a24.randomizeMin=-2;
		a24.randomizeMax=2;

		a25.caption="a25";
		a25.default=0;
		a25.randomizable=true;
		a25.min=-1E18;
		a25.max=1E18;
		a25.randomizeMin=-2;
		a25.randomizeMax=2;

		a26.caption="a26";
		a26.default=3;
		a26.randomizable=true;
		a26.min=-1E18;
		a26.max=1E18;
		a26.randomizeMin=-2;
		a26.randomizeMax=2;

		a27.caption="a27";
		a27.default=0;
		a27.randomizable=true;
		a27.min=-1E19;
		a27.max=1E19;
		a27.randomizeMin=-2;
		a27.randomizeMax=2;

		a28.caption="a28";
		a28.default=4;
		a28.randomizable=true;
		a28.min=-1E19;
		a28.max=1E19;
		a28.randomizeMin=-2;
		a28.randomizeMax=2;

		a29.caption="a29";
		a29.default=0;
		a29.randomizable=true;
		a29.min=-1E19;
		a29.max=1E19;
		a29.randomizeMin=-2;
		a29.randomizeMax=2;

		a30.caption="a30";
		a30.default=-4;
		a30.randomizable=true;
		a30.min=-1E20;
		a30.max=1E20;
		a30.randomizeMin=-2;
		a30.randomizeMax=2;

		a31.caption="a31";
		a31.default=1;
		a31.randomizable=true;
		a31.min=-1E20;
		a31.max=1E20;
		a31.randomizeMin=-2;
		a31.randomizeMax=2;

		a32.caption="a32";
		a32.default=0;
		a32.randomizable=true;
		a32.min=-1E20;
		a32.max=1E20;
		a32.randomizeMin=-2;
		a32.randomizeMax=2;

		a33.caption="a33";
		a33.default=-2;
		a33.randomizable=true;
		a33.min=-1E21;
		a33.max=1E21;
		a33.randomizeMin=-2;
		a33.randomizeMax=2;

		a34.caption="a34";
		a34.default=0;
		a34.randomizable=true;
		a34.min=-1E21;
		a34.max=1E21;
		a34.randomizeMin=-2;
		a34.randomizeMax=2;

		a35.caption="a35";
		a35.default=1;
		a35.randomizable=true;
		a35.min=-1E21;
		a35.max=1E21;
		a35.randomizeMin=-2;
		a35.randomizeMax=2;

	}
}

JeenerKleinSurface(ATTRACTOR)
{
real u,v;
real tx,ty,tz,W;
parameter real S,T,a;
parameter bool useInternalColor;
parameter real hBand,hBandSize,vBand,vBandSize;
shared double colorIndex;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=random.real();
		v=random.real();
		if (useInternalColor) {
			if (u*hBand-floor(u*hBand)<hBand*0.01*hBandSize || v*vBand-floor(v*vBand)<vBand*0.01*vBandSize) {
				colorIndex=0.5;
				// hiddenPixel=true;
			} else {
				colorIndex=0;
			}
		}
		

		u=u*2*pi;
		v=v*2*pi;
		W=((S + 1)/4)*cos((S + 1)*u + pi/T) + sqrt(a);

		tx = S*cos(u) + cos(S*u) - W*sin((S - 1)*u/2)*cos(v);
		ty = W*sin(v);
		tz = S*sin(u) - sin(S*u) - W*cos((S - 1)*u/2)*cos(v);

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Jeener-Klein Surface";

		S.caption="Num. Elements";
		S.default=5;
		S.randomizable=true;
		S.min=-1E10;
		S.max=1E10;
		S.randomizeMin=-5;
		S.randomizeMax=5;

		T.caption="T";
		T.default=4;
		T.randomizable=true;
		T.min=-1E10;
		T.max=1E10;
		T.randomizeMin=0;
		T.randomizeMax=5;

		a.caption="Inner Radius";
		a.default=2;
		a.randomizable=true;
		a.min=0;
		a.max=1E10;
		a.randomizeMin=0;
		a.randomizeMax=2;
		
		useInternalColor.caption="Internal Coloring";
		useInternalColor.default=true;
		useInternalColor.hint="If checked, the formula uses the internal coloring, which only works if you select coloring formula 'Attractor Internal Coloring'";
		
		hBand.caption="Horizontal Bands";
		hBand.hint="Specifies how many horizontal bands should appear";
		hBand.default=200;
		
		hBandSize.caption="Thickness";
		hBandSize.default=0.05;
		
		vBand.caption="Vertical Bands";
		vBand.hint="Specifies how many vertical bands should appear";
		vBand.default=30;

		vBandSize.caption="Thickness";
		vBandSize.default=0.1;
	}
}

BonanJeenerKlein(ATTRACTOR)
{
real u,v;
real tx,ty,tz,W;
parameter real M,T;
parameter bool useInternalColor;
parameter real hBand,hBandSize,vBand,vBandSize;
shared double colorIndex;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=random.real();
		v=random.real();
		if (useInternalColor) {
			if (u*hBand-floor(u*hBand)<hBand*0.01*hBandSize || v*vBand-floor(v*vBand)<vBand*0.01*vBandSize) {
				colorIndex=0.5;
				// hiddenPixel=true;
			} else {
				colorIndex=0;
			}
		}
		

		u=u*2*pi;
		v=v*2*pi;
		W=sin((M - 1)*u) + T;

		tx = M*cos(u) - cos(M*u) - ((M - 1)/M)*W*sin((M + 1)*u/2)*cos(v);
		ty = W*sin(v);
		tz = M*sin(u) - sin(M*u) + ((M - 1)/M)*W*cos((M + 1)*u/2)*cos(v);

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Bonan-Jeener-Klein Surface";

		M.caption="Num. Elements";
		M.default=3;
		M.randomizable=true;
		M.min=-1E10;
		M.max=1E10;
		M.randomizeMin=-5;
		M.randomizeMax=5;

		T.caption="T";
		T.default=1.5;
		T.randomizable=true;
		T.min=-1E10;
		T.max=1E10;
		T.randomizeMin=0;
		T.randomizeMax=5;

		useInternalColor.caption="Internal Coloring";
		useInternalColor.default=true;
		useInternalColor.hint="If checked, the formula uses the internal coloring, which only works if you select coloring formula 'Attractor Internal Coloring'";
		
		hBand.caption="Horizontal Bands";
		hBand.hint="Specifies how many horizontal bands should appear";
		hBand.default=100;
		
		hBandSize.caption="Thickness";
		hBandSize.default=0.1;
		
		vBand.caption="Vertical Bands";
		vBand.hint="Specifies how many vertical bands should appear";
		vBand.default=30;

		vBandSize.caption="Thickness";
		vBandSize.default=0.2;
	}
}

BraidedTorus(ATTRACTOR)
{
real u,v;
real tx,ty,tz;
parameter real a,n,R,r;
parameter bool useInternalColor;
parameter real hBand,hBandSize,vBand,vBandSize;
shared double colorIndex;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=random.real();
		v=random.real();
		if (useInternalColor) {
			if (u*hBand-floor(u*hBand)<hBand*0.01*hBandSize || v*vBand-floor(v*vBand)<vBand*0.01*vBandSize) {
				colorIndex=0.5;
				// hiddenPixel=true;
			} else {
				colorIndex=0;
			}
		}
		

		u=u*8*pi;
		v=v*2*pi;

		tx = r*cos(v)*cos(u) + R*cos(u)*(1 + a*cos(n*u));
		ty = 2.5*(r*sin(v) + a*sin(n*u));
		tz = r*cos(v)*sin(u) + R*sin(u)*(1 + a*cos(n*u));

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Braided Torus";

		a.caption="a";
		a.default=0.5;
		a.randomizable=true;
		a.min=-1E10;
		a.max=1E10;
		a.randomizeMin=-5;
		a.randomizeMax=5;

		n.caption="n";
		n.default=1.25;
		n.randomizable=true;
		n.min=-1E10;
		n.max=1E10;
		n.randomizeMin=-5;
		n.randomizeMax=5;

		R.caption="Overall Size";
		R.default=2;
		R.randomizable=true;
		R.min=-1E10;
		R.max=1E10;
		R.randomizeMin=-5;
		R.randomizeMax=5;

		r.caption="Thickness";
		r.default=0.35;
		r.randomizable=true;
		r.min=-1E10;
		r.max=1E10;
		r.randomizeMin=-5;
		r.randomizeMax=5;

		useInternalColor.caption="Internal Coloring";
		useInternalColor.default=true;
		useInternalColor.hint="If checked, the formula uses the internal coloring, which only works if you select coloring formula 'Attractor Internal Coloring'";
		
		hBand.caption="Horizontal Bands";
		hBand.hint="Specifies how many horizontal bands should appear";
		hBand.default=200;
		
		hBandSize.caption="Thickness";
		hBandSize.default=0.05;
		
		vBand.caption="Vertical Bands";
		vBand.hint="Specifies how many vertical bands should appear";
		vBand.default=20;

		vBandSize.caption="Thickness";
		vBandSize.default=0.5;
	}
}

KleinCycloid(ATTRACTOR)
{
real u,v;
real tx,ty,tz;
parameter real a,b,c;
parameter bool useInternalColor;
parameter real hBand,hBandSize,vBand,vBandSize;
shared double colorIndex;

	void init(void)
	{
		// empty...
	}
	void loop(void)
	{
		u=random.real();
		v=random.real();
		if (useInternalColor) {
			if (u*hBand-floor(u*hBand)<hBand*0.01*hBandSize || v*vBand-floor(v*vBand)<vBand*0.01*vBandSize) {
				colorIndex=0.5;
				// hiddenPixel=true;
			} else {
				colorIndex=0;
			}
		}
		

		u=u*2*b*c*pi;
		v=v*2*pi;

		tx = cos(u/c) * cos(u/b) * (a + cos(v)) + sin(u/b) * sin(v) * cos(v);
		ty = sin(u/c) * cos(u/b) * (a + cos(v)) + sin(u/b) * sin(v) * cos(v);
		tz = -sin(u/b) * (a + cos(v)) + cos(u/b) * sin(v) * cos(v);

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Klein Cycloid";

		a.caption="a";
		a.default=10;
		a.randomizable=true;
		a.min=-1E10;
		a.max=1E10;
		a.randomizeMin=-10;
		a.randomizeMax=10;

		b.caption="b";
		b.default=3;
		b.randomizable=true;
		b.min=-1E10;
		b.max=1E10;
		b.randomizeMin=-5;
		b.randomizeMax=5;

		c.caption="c";
		c.default=2;
		c.randomizable=true;
		c.min=-1E10;
		c.max=1E10;
		c.randomizeMin=-5;
		c.randomizeMax=5;

		useInternalColor.caption="Internal Coloring";
		useInternalColor.default=true;
		useInternalColor.hint="If checked, the formula uses the internal coloring, which only works if you select coloring formula 'Attractor Internal Coloring'";
		
		hBand.caption="Horizontal Bands";
		hBand.hint="Specifies how many horizontal bands should appear";
		hBand.default=300;
		
		hBandSize.caption="Thickness";
		hBandSize.default=0.05;
		
		vBand.caption="Vertical Bands";
		vBand.hint="Specifies how many vertical bands should appear";
		vBand.default=20;

		vBandSize.caption="Thickness";
		vBandSize.default=1;
	}
}


Spirograph(ATTRACTOR)
{
real u,maxRange;
real tx,ty,tz;
parameter real R,r,a,rounds;
parameter int type;

	void init(void)
	{
		maxRange=2*pi*R/r*rounds;
	}
	void loop(void)
	{
		u=random.real()*maxRange;
		if (type=="Hypozykloid") {
			tx=(R-r)*cos(u*r/R)+a*cos(u*(1-r/R));
			ty=(R-r)*sin(u*r/R)-a*sin(u*(1-r/R));
		} else if (type=="Epizykloid") {
			tx=(R+r)*cos(u*r/R)-a*cos(u*(1+r/R));
			ty=(R+r)*sin(u*r/R)-a*sin(u*(1+r/R));
		}
		tz=0;

		z=vector(tx,ty,tz);
	}
	real scale(void)
	{
		return 0.5;
	}
	void description(void)
	{
		this.title="Spirograph";
		
		type.caption="Type";
		type.enum="Hypozykloid\nEpizykloid";
		type.default=0;

		R.caption="Outer Radius";
		R.hint="Outer Radius of Spirograph";
		R.default=5;
		R.min=0;
		R.max=1E10;
		R.randomizable=true;
		R.randomizeMin=1;
		R.randomizeMax=10;
		
		r.caption="Inner Radius";
		r.hint="Inner Radius of Spirograph";
		r.default=1.3;
		r.min=-1E10;
		r.max=1E10;
		r.randomizable=true;
		r.randomizeMin=-10;
		r.randomizeMax=10;
		
		a.caption="Eccentric";
		a.hint="Eccentricity of inner cirle of spirograph";
		a.default=3;
		a.min=0;
		a.max=1E10;
		a.randomizable=true;
		a.randomizeMin=0;
		a.randomizeMax=10;
		
		rounds.caption="Rounds";
		rounds.hint="Number of rounds around outer circle. Graph may be closed or not...";
		rounds.default=10;
		rounds.min=0;
		rounds.max=1E10;
	}
}
